﻿using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Collections.Generic;
using System.Linq;
using System.ServiceModel;
using System.Text;

namespace VA.PPMS.CRM.Plugins
{
    public class ProviderChangeNpi : IPlugin
    {
        private const string PluginName = "ProviderChangeNpi";
        private const string MessageCreate = "CREATE";
        private const string MessageUpdate = "UPDATE";

        public void Execute(IServiceProvider serviceProvider)
        {
            // Tracing service for debugging
            ITracingService tracingService = (ITracingService)serviceProvider.GetService(typeof(ITracingService));

            // Get execution context
            IPluginExecutionContext context = (IPluginExecutionContext)serviceProvider.GetService(typeof(IPluginExecutionContext));

            if (context.InputParameters.Contains("Target") && context.InputParameters["Target"] is Entity)
            {
                tracingService.Trace("Begin");

                // Obtain the target entity from the input parameters.
                Entity entity = (Entity)context.InputParameters["Target"];

                // Verify target entity type
                if (entity.LogicalName != "account")
                    return;

                tracingService.Trace("Entity found");

                // Get organization service reference
                IOrganizationServiceFactory serviceFactory = (IOrganizationServiceFactory)serviceProvider.GetService(typeof(IOrganizationServiceFactory));
                IOrganizationService service = serviceFactory.CreateOrganizationService(context.UserId);

                try
                {
                    bool delete = false;
                    bool match = false;
                    // handle event based on message type
                    switch (context.MessageName.ToUpper())
                    {
                        case MessageCreate:
                            // Check for exclusion by demographics
                            var result = GetMatchingExclusionsByComposite(service, entity);
                            if (result != null && result.Entities != null && result.Entities.Any())
                            {
                                tracingService.Trace("Composite deactivate request");
                                delete = false;
                                match = true;
                            }
                            break;

                        case MessageUpdate:
                            tracingService.Trace("Retrieve NPI");
                            string npi = GetNpi(entity, service);
                            if (!String.IsNullOrEmpty(npi))
                            {
                                tracingService.Trace("NPI found = {0}", npi);

                                // Check for exclusions by NPI
                                tracingService.Trace("Retrieve entity");
                                var npiExclusions = GetMatchingExclusionsByNpi(service, npi);
                                if (npiExclusions != null && npiExclusions.Entities.Count > 0)
                                {
                                    tracingService.Trace("NPI deactivate request");
                                    delete = true;
                                    match = true;
                                }
                            }
                            break;

                        default:
                            tracingService.Trace("Valid event message not found");
                            break;
                    }

                    if (delete)
                    {
                        tracingService.Trace("Deactivate provider...");
                        SetStateRequest request = PpmsHelper.GetDeactivateRequest(entity, (int)PpmsHelper.Account_StatusCode.LeieExclusion);
                        service.Execute(request);
                    }

                    if (match)
                    {
                        tracingService.Trace("Create history log...");
                        SetStateRequest request = PpmsHelper.GetDeactivateRequest(entity, (int)PpmsHelper.Account_StatusCode.LeieExclusion);
                        service.Execute(request);
                    }

                    // Create LEIE validation entry
                    tracingService.Trace("Create provider validation entries...");
                    PpmsHelper.AddProviderValidation(service, entity.ToEntityReference(), PpmsHelper.Validation_Type.Leie);
                }
                catch (FaultException<OrganizationServiceFault> ex)
                {
                    tracingService.Trace("Fault: {0}", ex.ToString());
                    throw new InvalidPluginExecutionException(String.Format("An error occurred in {0}.", PluginName), ex);
                }
                catch (Exception ex)
                {
                    tracingService.Trace("Exception: {0}", ex.ToString());
                    throw;
                }
            }
            tracingService.Trace("Done");
        }

        private string GetNpi(Entity provider, IOrganizationService service)
        {
            var entity = provider.GetAttributeValue<EntityReference>("ppms_npiid");
            if (entity != null)
            {
                var npi = service.Retrieve("ppms_provideridentifier", entity.Id, new ColumnSet(new string[] { "ppms_provideridentifier" }));
                return npi.GetAttributeValue<string>("ppms_provideridentifier");
            }

            return null;
        }

        private EntityCollection GetMatchingExclusionsByNpi(IOrganizationService service, string npi)
        {
            FilterExpression filter = new FilterExpression();
            filter.AddCondition("ppms_npi", ConditionOperator.Equal, npi);
            filter.AddCondition("ppms_action", ConditionOperator.Equal, (int)PpmsHelper.LeieAction.Exclude);
            filter.AddCondition("statecode", ConditionOperator.Equal, (int)PpmsHelper.AccountState.Active);

            QueryExpression query = new QueryExpression("ppms_leieexclusion");
            query.ColumnSet.AddColumns("ppms_npi", "ppms_action");
            query.Criteria.AddFilter(filter);

            return service.RetrieveMultiple(query);
        }

        private EntityCollection GetMatchingExclusionsByComposite(IOrganizationService service, Entity provider)
        {
            // Get provider properties
            var dob = provider.GetAttributeValue<DateTime?>("ppms_indivproviderdateofbirth");
            string composite = PpmsHelper.GetProviderComposite(provider);

            FilterExpression filter = new FilterExpression();
            filter.AddCondition("statecode", ConditionOperator.Equal, (int)PpmsHelper.AccountState.Active);
            filter.AddCondition("ppms_action", ConditionOperator.Equal, (int)PpmsHelper.LeieAction.Exclude);
            filter.AddCondition("ppms_composite", ConditionOperator.Like, composite);

            // Check for DOB
            if (dob.HasValue) filter.AddCondition("ppms_dob", ConditionOperator.Between, dob.Value, dob.Value.AddDays(1));

            QueryExpression query = new QueryExpression("ppms_leieexclusion");
            query.ColumnSet.AddColumns("ppms_npi", "ppms_action");
            query.Criteria.AddFilter(filter);

            return service.RetrieveMultiple(query);
        }
    }
}
